/*
 * Copyright 2019 NXP
 * All rights reserved.
 * SPDX-License-Identifier: BSD-3-Clause
 */

/*!
 * @addtogroup GENFSK_Localization
 * @addtogroup PhaseBaseDistEstimate Phase-based Distance Estimation localization feature
 * @ingroup GENFSK_Localization
 * @{
 */

/*! @file
 * GENFSK Localization - Phase-based Distance Estimation using complex domain estimation (CDE).
 */

#include <stdio.h>
#include "arm_math.h"

#ifndef DM_PHASEBASED_H

#define DM_PHASEBASED_H

/*******************************************************************************
 * Definitions
 ******************************************************************************/

/* Constants */
#define C_LIGHT    (300000000)
#define Q10_SCALING_FACTOR    (1024)
#define Q14_SCALING_FACTOR    (16384)
#define Q15_SCALING_FACTOR    (32768)

#define DM_MAX_FREQ_NB (256U) /*!< Maximum number of frequencies that can be passed to the API */
#define N_FFT_MAX  (256)      /*!< Maximum size of the FFT that can be requested to the API */

/*******************************************************************************
 * Types
 ******************************************************************************/

/*!
 * API return codes
 */
typedef enum {
    DM_STATUS_OK,
    DM_STATUS_INVALID_PARAMETER,
} dm_status_t;

/*!
 * @brief Distance measurement output for CDE method.
 */
typedef struct
{
    int              distance;          /*!< Distance computed by the present algorithm in Q21.10 format */
    int              distance_interp;   /*!< Distance computed by the interpolation variant of present algorithm in Q21.10 format */
    unsigned short   d_fft_idx;         /*!< IFFT index chosen by the algorithm. Range is [0; N_FFT-1]. */
    q15_t            dqi;               /*!< Distance quality indicator metric in Q2.14 format. */
    unsigned char    threshold_reached; /*!< Reserved. */
} dm_cde_out_t;

/*!
 * @brief Distance measurement output for CDE method.
 */
typedef struct
{
    int              distance;          /*!< Distance computed by the present algorithm in Q16.15 format */
    q15_t            dqi;               /*!< Distance quality indicator metric in Q15 format. */
} dm_slope_out_t;

/*!
 * @brief Distance measurement debug information output for CDE method.
 */
typedef struct
{
    q15_t            *fft_mag_p;        /*!< Pointer to IFFT magnitude array. Values are Q2.14. Array size is N_FFT. */
} dm_cde_debug_info_t;

/*!
 * @brief Distance measurement debug information output for slope-based method.
 */
typedef struct
{
    q15_t            slope;             /*!< Computed slope in Q15. */
} dm_slope_debug_info_t;

/*!
 * @brief Frequency validity mask.
 * mask[0] : b31 (MSB) ............... b0 (LSB)
 * mask[0] : Fstart + 31*freq_step ... Fstart
 * etc...
 * '0' means that phase value for corresponding frequency in input vector is invalid
 * '1' means that phase value for corresponding frequency in input vector is valid
 * unused frequency must be padded with zeros
 */
typedef struct
{
    uint32_t         mask[DM_MAX_FREQ_NB/32];
} dm_freq_mask_t;

/*******************************************************************************
 * Prototypes
 ******************************************************************************/

/*!
 * @brief Phase based distance measurement algorithm.
 *
 * Use complex-domain estimation.
 *
 * @param nb_meas          Number of entries in phi_fp vector, corresponding to the number of used frequencies (input)
 * @param phi_fp           Pointer on Phi vector array as defined in the theory. (input)
                           Phase values (phi_fp[]) are expected to be scaled to [-1;1[ in Q15 fixed-point format
 * @param freq_mask        Pointer on Frequency validity mask (input)
 * @param freq_step        Frequency step in KHz between 2 consecutive frequencies (input)
 * @param n_fft            IFFT size. Allowed values are 64, 128 and 256 (recommended) (input)
 * @param threshold        IFFT magnitude threshold in 0.001 steps. Typical value is 200 (0.2) - reserved, set to zero
 * @param nb_est           Number of distance estimations required from the algorithm. Also size of out_p as provided by the caller.
 * @param out_p            Pointer on array of estimate (size nb_est)
 * @param dbg_info_p       Pointer on additional debug info structure. May be NULL. (output)
 *
 * @retval dm_status_t
 *
 */
dm_status_t dm_complex_domain_based_fp(const unsigned short nb_meas,
                                       const q15_t phi_fp[],
                                       const dm_freq_mask_t *freq_mask,
                                       const unsigned short freq_step,
                                       const unsigned short n_fft,
                                       const unsigned short threshold,
                                       const unsigned char  nb_est,
                                       dm_cde_out_t         out_p[],
                                       dm_cde_debug_info_t *dbg_info_p
                                       );

/*!
 * @brief Phase based distance measurement algorithm.
 *
 * Slope-based estimation:
 *
 * @param nb_meas          Number of entries in phi_fp vector, corresponding to the number of used frequencies (input)
 * @param phi_fp           Pointer on Phi vector array as defined in the theory. (input)
                           Phase values (phi_fp[]) are expected to be scaled to [-1;1[ in Q15 fixed-point format
 * @param freq_mask        Pointer on Frequency validity mask (input)
 * @param freq_step_khz    Frequency step in KHz between 2 consecutive frequencies (input)
 * @param out_p            Pointer on structure for storing estimation result.
 * @param dbg_info_p       Pointer on additional debug info structure. May be NULL. (output)
 *
 * @retval dm_status_t
 *
 */
dm_status_t dm_slope_based_fp(const unsigned short nb_meas,
                              const q15_t phi_fp[],
                              const dm_freq_mask_t *freq_mask,
                              const unsigned short f_step_khz,
                              dm_slope_out_t *out_p,
                              dm_slope_debug_info_t *dbg_info_p
                              );

#endif /* DM_PHASEBASED_H */

/*! @} */
